import java.util.logging.*;

/**
 * Vincenty's formulae を使って大圏航路を計算するクラス。
 *
 * @author Toshi Tamamori
 */

public class VincentyFormulae implements GreatArcAlg {
	
	private final double EPSILON  = 0.000000000001; // 収束判定のための微小値
	private final int    MAX_ITER = 20;             // 反復の上限値
	
	/**
	 * コンストラクタでは何もしません。
	 */
	public VincentyFormulae() { }
	
	// DIRECT問題
	// 引数
	//   準拠地球楕円体（具象クラスそのものではなく、抽象クラス（インタフェース）で受け取る
	//   始点の位置（緯度・経度。単位：ラジアン）
	//   到達点への距離（単位：メートル）
	//   始点からの方位角（単位：ラジアン）
	//
	public DirectResult direct(EarthEllipsoid ellipsoid, LatLng source,
	                           double distance, double azimuth)        {
		
		Logger logger = GreatArcLogger.getLogger();
		logger.log(Level.FINER, "direct() started.");
		
		/* 変数名は、極力 Vincentyの論文のものに合わせた。
		 * 式が煩雑かつ長文となり読みにくくならないように、
		 * できるだけ短く、かつ、ワーク用の変数も使用している。
		 */
		double a	= ellipsoid.getAxis();
		double f	= ellipsoid.getFlattening();			// 扁平率 = (a-b)/a
		double b	= (1.0-f) * a;
		double phi1	= source.getLatRad();					// １点目（始点）の緯度
		double phi2	; // unknown.							// ２点目（到達点）の緯度
		double U1	= Math.atan((1.0-f) * Math.tan(phi1));	// 定義(reduced緯度)
		// double U2	; // not used in direct()
		double L	; // unknown.							// ２点間の経度差（東向きが正)
		double L1	= source.getLngRad();					// １点目（始点）の経度
		double L2	; // unknown.							// ２点目（到達点）の経度
		
		double alp1	= azimuth;								// １点目（始点）からの方位角
		double alp2	; // unknown							// ２点目（到達点）からの方位角
		double s	= distance;								// 距離
		double sigma;										// ２点間の球体上での角距離
		double sig1 ;										// 赤道から１点目（始点）への角距離
		
		double sinAlpha;			// sin(alpha)
		double cosAlphaSquare;		// (cos(alpha))^2
		double uSquare;				// u^2
		double A;
		double B;
		double sigmx2;										// midpoint sigmaの２倍(2 * sigma_mid)を短縮表現した変数名とした
		
		/* 反復計算前のいくつかの変数の初期化 */
		sig1   = Math.atan2(Math.tan(U1), Math.cos(alp1));	// （式１） tan(sig1) = tan(U1)/cos(alp1)
		sinAlpha = Math.cos(U1) * Math.sin(alp1);			// （式２） sin(alpha) = cos(U1)*sin(alp1)
		cosAlphaSquare = 1.0 - (sinAlpha * sinAlpha);       //  (cos(alpha))^2 = 1 - (sin(alpha))^2
		uSquare   = cosAlphaSquare * (a * a - b * b) / (b * b);    // u^2 = (cos(alpha))^2 * (a^2 - b^2) / b^2
		
		A = 1.0 + (4096.0 + uSquare * (-768.0 + uSquare * (320.0 - 175.0 * uSquare))) * uSquare / 16384.0; // （式３）
		B = (256.0 + uSquare * (-128.0 + uSquare * (74.0 - 47.0 * uSquare))) * uSquare / 1024.0; // （式４）
		
		sigma = s / (b * A); 								// （式７）　反復前のsigmaの初期値
		
		/* sigmaが収束するまで反復。
		   loop はループ回数(反復回数)を収束しない場合のために使用する。
		 */
		for(int loop=1; ;loop++) {
			
			logger.log(Level.FINEST, "loop=" + loop + " sigma=" + sigma);
			
			double delta;
			double sinSigma = Math.sin(sigma);
			double cosSigmx2;
			
			sigmx2 = 2.0 * sig1 + sigma;				// （式５）
			cosSigmx2 = Math.cos(sigmx2);				// 下記で多用するので、一旦変数として保持する。　
			
			double work1 = (-3.0 + 4.0 * sinSigma * sinSigma);
			double work2 = (-3.0 + 4.0 * cosSigmx2 * cosSigmx2);
			double work3 = B * cosSigmx2 * work1 * work2 / 6.0;
			double work4 = B * (Math.cos(sigma) * (-1.0 + 2.0 * cosSigmx2 * cosSigmx2) - work3) /4.0;
			
			delta = B * sinSigma * (cosSigmx2 + work4 ); 	// (式６）
			
			double sigmaPrev = sigma;
			sigma =  s / (b * A) + delta;								// （式７）
			
			logger.log(Level.FINEST, "loop=" + loop + " delta=" + delta);
			
			// 収束したらループを抜ける
			if(Math.abs(sigma - sigmaPrev) < EPSILON) { break; }
			
			// 反復条件を超えて収束しなかったら諦める
			if(loop > MAX_ITER) {
				logger.log(Level.WARNING, "direct method not converged. loop=" + loop);
				break;
			}
			
		}
		
		/* 収束したので、解を求める */
		{
			double sinU1    = Math.sin(U1);
			double cosU1    = Math.cos(U1);
			double sinSigma = Math.sin(sigma);
			double cosSigma = Math.cos(sigma);
			double cosAlp1  = Math.cos(alp1);
			
			double work1 = sinU1 * cosSigma + cosU1 * sinSigma * cosAlp1;
			double work2 = sinU1 * sinSigma - cosU1 * cosSigma * cosAlp1;
			double work3 = Math.sqrt(sinAlpha * sinAlpha + work2 * work2);
			double work4 = (1.0 - f) * work3;
			
			phi2 = Math.atan2(work1, work4);	// （式８）
			
			double work5 = sinSigma * Math.sin(alp1);
			double work6 = cosU1 * cosSigma - sinU1 * sinSigma * Math.cos(alp1);
			
			double lambda = Math.atan2(work5, work6);	// （式９）
			
			double C = f * cosAlphaSquare * (4.0 + f * (4.0 - 3.0 * cosAlphaSquare)) / 16.0;	// （式１０）
			
			double work7 = Math.cos(sigmx2);
			double work8 = C * cosSigma * (-1.0 + 2.0 * work7 * work7);
			double work9 = sigma + C * sinSigma * (work7 + work8);
			
			L  = lambda - (1.0 - C) * f * sinAlpha * work9;		// （式１１）
			L2 = L + L1;
			
			// 経度が180度（πラジアン）を超えたら、日付変更線を跨いだということ
			if(Math.PI < L2)  { L2 -= Math.PI * 2.0; }
			if(L2 < -Math.PI) { L2 += Math.PI * 2.0; }
			
			alp2 = Math.atan2(sinAlpha, (cosU1 * cosSigma * cosAlp1 - sinU1 * sinSigma));	// （式１２）
			alp2 += Math.PI; // 式１２が何かおかしいのかな？？？
		}
		
		// 戻り値のオブジェクトに計算結果を設定して戻す
		// ここって、１行で書けないのかな？
		LatLng        pointReached = new LatLng(Math.toDegrees(phi2), Math.toDegrees(L2));
		DirectResult  ret = new DirectResult();
		
		ret.pointReached = pointReached;
		ret.azimuthFromPoint2 = alp2;
		
		logger.log(Level.FINER, "direct() terminated normally.");
		
		return ret;
	}
	
	// INVERSE問題
	// 引数
	//   準拠地球楕円体（具象クラスそのものではなく、抽象クラス（インタフェース）で受け取る
	//   始点の位置（緯度・経度。単位：ラジアン）
	//   到達点位置（緯度・経度。単位：ラジアン）
	//
	public InverseResult inverse(EarthEllipsoid ellipsoid, LatLng point1, LatLng point2) {
		
		Logger logger = GreatArcLogger.getLogger();
		logger.log(Level.FINER, "inverse() started.");
		
		/* 変数名は、極力 Vincentyの論文のものに合わせた。
		 * 式が煩雑かつ長文となり読みにくくならないように、
		 * できるだけ短く、かつ、ワーク用の変数も使用している。
		 */
		double a	= ellipsoid.getAxis();
		double f	= ellipsoid.getFlattening();
		double b	= (1.0-f) * a;
		double phi1	= point1.getLatRad();
		double phi2	= point2.getLatRad();
		double U1	= Math.atan((1.0-f) * Math.tan(phi1));
		double U2	= Math.atan((1.0-f) * Math.tan(phi2));
		double L1	= point1.getLngRad();
		double L2	= point2.getLngRad();
		double L	= L2 - L1;
		double lambda;
		double sinLambda;
		double cosLambda;
		double alp1	; // unknown　始点から終点への方位角（ラジアン）
		double alp2	; // unknown　終点から始点への方位角（ラジアン）
		double s	; // unknown　始点と終点との間の距離（メートル）
		double sigma;
		double sinSigma;
		double cosSigma;
		
		double sinU1 = Math.sin(U1);
		double cosU1 = Math.cos(U1);
		double sinU2 = Math.sin(U2);
		double cosU2 = Math.cos(U2);
		
		double cosAlphaSquare;			// (cos(alpha))^2
		double uSquare;					// u^2
		double cosSigmx2;				// cos(2sigm)
		
		/* 反復前の必要な初期化 (初期値は経度差だけの角距離）*/
		lambda = L;						// （式１３）
		
		/* 収束するまで反復 */
		for(int loop=1; ;loop++) {
			
			logger.log(Level.FINEST, "loop=" + loop + " lambda=" + lambda);
			
			double lambdaPrev = lambda;
			sinLambda = Math.sin(lambda);
			cosLambda = Math.cos(lambda);
			logger.log(Level.FINEST, "sinλ=" + sinLambda + " cosλ=" + cosLambda);
			
			double work1 = cosU2 * cosU2 * sinLambda * sinLambda;
			double work2 = cosU1 * sinU2 - sinU1 * cosU2 * cosLambda;
			double work3 = work2 * work2;
			
			// sin(sigma)が0の場合は、２点が同一店または対蹠点
			sinSigma = Math.sqrt(work1 + work3);					// （式１４）
			cosSigma = sinU1 * sinU2 + cosU1 * cosU2 * cosLambda;	// （式１５）
			logger.log(Level.FINEST, "sinσ=" + sinSigma + " cosσ=" + cosSigma);
			
			sigma = Math.atan2(sinSigma, cosSigma);					// （式１６）
			
			double sinAlpha = cosU1 * cosU2 * sinLambda / sinSigma;	// （式１７）
			cosAlphaSquare = 1.0 - (sinAlpha * sinAlpha);
			
			cosSigmx2 = cosSigma - (2.0 * sinU1 * sinU2) / cosAlphaSquare;	// （式１８）
			logger.log(Level.FINEST, "cos(2σm)=" + cosSigmx2);
			
			double C = f * cosAlphaSquare * (4.0 + f * (4.0 - 3.0 * cosAlphaSquare)) / 16.0;	// （式１０）
			
			double work5 = C * cosSigma * (-1.0 + 2.0 * cosSigmx2 * cosSigmx2);
			double work6 = C * sinSigma * (cosSigmx2 + work5);
			
			lambda = L + (1.0 - C) * f * sinAlpha * (sigma + work6); // （式１１）
			logger.log(Level.FINEST, "lambda=" + lambda);
			
			// 収束したらループ（反復）を抜ける
			if (Math.abs(lambda - lambdaPrev) < EPSILON) { break; }
			
			// 反復条件を超えて収束しなかったら諦める
			if(loop > MAX_ITER) {
				logger.log(Level.WARNING, "inverse method not converged. loop=" + loop);
				break;
			}
		}
		
		// 反復後の計算
		{
			uSquare = cosAlphaSquare * (a * a - b * b) / (b * b);
			
			double work7 = uSquare * (320.0 - 175.0 * uSquare);
			double work8 = uSquare * (-768.0 + work7);
			double A = 1.0 + uSquare * (4096.0 + work8) / 16384.0;			// （式３）
			
			double work9  = uSquare * (74.0 - 47.0 * uSquare);
			double work10 = uSquare * (-128.0 + work9);
			double B = uSquare * (256.0 + work10) / 1024.0;					// （式４）
			
			double work11 = -3.0 + 4.0 * cosSigmx2 * cosSigmx2;
			double work12 = -3.0 + 4.0 * sinSigma * sinSigma;
			double work13 = B * cosSigmx2 * work12 * work11 / 6.0;
			double work14 = -1.0 + 2.0 * cosSigmx2 * cosSigmx2;
			double work15 = B * (cosSigma * work14 - work13) / 4.0;
			
			double delta = B * sinSigma * (cosSigmx2 + work15);				// (式６）
			
			s = b * A * (sigma - delta);				// （式１９）　（式７の逆算）
			
			double work16 = cosU2 * sinLambda;
			double work17 = cosU1 * sinU2 - sinU1 * cosU2 * cosLambda;
			
			alp1 = Math.atan2(work16, work17);			// （式２０）
			if(alp1 < 0.0) { alp1 += Math.PI * 2.0; }
			
			double work18 = cosU1 * sinLambda;
			double work19 = cosU1 * sinU2 * cosLambda - sinU1 * cosU2;
			
			alp2 = Math.atan2(work18, work19);			// （式２１）
			alp2 += Math.PI;
		}
		
		// 値を入れて返却する。
		InverseResult ret = new InverseResult();
		ret.distance          = s;
		ret.azimuthFromPoint1 = alp1;
		ret.azimuthFromPoint2 = alp2;
		
		logger.log(Level.FINER, "inverse() terminated normally.");
		
		return ret;
	}
}