import java.util.logging.*;

/**
 * 球面三角法(Spherical Trigonometry) を使った大圏航路計算クラス。
 *
 * 仮に、真球をあたえられたかったとしても、扁平率は使わず与えられた半径だけを
 * 使って計算する。
 *
 * @author Toshi Tamamori
 */
//
// 北極をA、１点目をB、２点目をC　ととって、計算する。
//

public class SphericalTrigo implements GreatArcAlg {
	
	/**
	 * コンストラクタでは何もしません。
	 */
	public SphericalTrigo() { }
	
	// DIRECT問題
	// 引数
	//   準拠地球楕円体（具象クラスそのものではなく、抽象クラス（インタフェース）で受け取る
	//   始点の位置（緯度・経度。単位：ラジアン）
	//   到達点への距離（単位：メートル）
	//   始点からの方位角（単位：ラジアン）
	//
	public DirectResult direct(EarthEllipsoid ellipsoid, LatLng source,
	                           double distance, double azimuth)        {
		
		Logger logger = GreatArcLogger.getLogger();
		logger.log(Level.FINER, "direct() started.");
		
		double	lat1 = source.getLatRad();	// 始点の緯度（ラジアン）
		double	lng1 = source.getLngRad();	// 始点の経度（ラジアン）
		double	lat2;						// unknown 到達点の緯度（ラジアン）
		double	lng2;						// unknown 到達点の経度（ラジアン）
		double	A;							// unknown 始点から到達点への経度差（東向きが正）
		double	B = azimuth;				// 始点から到達点へ向けた方位角（北が０。時計回りが正）
		double	C;							// 到達点から始点へ向けた方位角（北が０。時計回りが正）
		double	a = distance / ellipsoid.getAxis();		// 始点と到達点と中心角（半径を掛けると距離となる）
		double	b;							// 北極と到達点との中心角（π／２　から到達点の緯度を減算したものに一致）
		double	c = Math.PI / 2.0 - lat1;	// 北極と始点との中心角（π／２　から始点の緯度を減算したものに一致）
		
		// 扁平率が０以外のモデルが与えられた場合は、ワーニングを出す。
		{
			if(ellipsoid.getFlattening() != 0.0) {
				logger.log(Level.WARNING, "flattening of the given speroid is NOT 0.0");
			}
		}
		
		// まず到達点の緯度を求める
		// cos(b) = cos(c)*cos(a) + sin(c)*sin(a)+cos(B) から b を求める。
		//
		{
			b = Math.acos(Math.cos(c) * Math.cos(a) + Math.sin(c) * Math.sin(a) * Math.cos(B));
			lat2 = Math.PI / 2.0 - b;
			logger.log(Level.FINER, "lat2(deg) = " + Math.toDegrees(lat2));
		}
		
		// 次に到達点の経度を求める
		// cos(a) = cos(b)*cos(c) + sin(b)*sin(c)*cos(A) からAを求めるのでは上手くいかない。
		// acos()は、０～π（正値）しか返さず、経度の減算に対応できないから。
		// lon2-lon1 = atan2( sin(B) * sin(a) * sin(c) / ( cos(a) - cos(b) * cos(c) ) )
		// 
		{
			double work = Math.atan2( Math.sin(B) * Math.sin(a) * Math.sin(c),
			                          Math.cos(a) - Math.cos(b) * Math.cos(c) );
			lng2 = lng1 + work;
			logger.log(Level.FINER, "lng2(deg) = " + Math.toDegrees(lng2));
			
			// 日付変更線を超えた結果が出たら360度を加算または減算する
			if(lng2 >  Math.PI) { lng2 -= Math.PI * 2.0; }
			if(lng2 < -Math.PI) { lng2 += Math.PI * 2.0; }
			logger.log(Level.FINER, "lng2(deg) = " + Math.toDegrees(lng2));
		}
		
		// 到達点から始点への方位角を求める
		// cos(c) = cos(a)*cos(b) + sin(a)*sin(b)+cos(C) からCを求める。
		{
			C = Math.acos(  (Math.cos(c) - Math.cos(a) * Math.cos(b))
			              / (Math.sin(a) * Math.sin(b))               );
			
			logger.log(Level.FINER, "az2to1(deg) = " + Math.toDegrees(C));
			
			// 西向きならば是正
			// これだけじゃ足りない。。。acos()は、０～π（北～東～南）を返し、
			// π～２π（南～西～北）が分からないから。
			// 経度差がプラスで１８０°を超えていたら、実は始点から西向き
			//                           超えていなければ、到達点から西向き
			// 経度差がマイナスで－１８０°を超えていたら、実は到達点から西向き
			//                               超えていなければ、始点から西向き
			A = lng2 - lng1;
			
			if(A >= 0.0) {
				if(A >= Math.PI)  {                      ; }
				else              { C = Math.PI * 2.0 - C; }
			} else {
				if(A <= -Math.PI) { C = Math.PI * 2.0 - C; }
				else              {                      ; }
			}
			
			logger.log(Level.FINER, "az2to1(deg) = " + Math.toDegrees(C));
		}
		
		// 戻り値のオブジェクトに計算結果を設定して戻す
		// ここって、１行で書けないのかな？
		LatLng       pointReached = new LatLng(Math.toDegrees(lat2), Math.toDegrees(lng2));
		DirectResult ret          = new DirectResult();
		ret.pointReached      = pointReached;
		ret.azimuthFromPoint2 = C;
		
		logger.log(Level.FINER, "direct() tarminated normally.");
		return ret;
	}

	// INVERSE問題
	// 引数
	//   準拠地球楕円体（具象クラスそのものではなく、抽象クラス（インタフェース）で受け取る
	//   始点の位置（緯度・経度。単位：ラジアン）
	//   到達点位置（緯度・経度。単位：ラジアン）
	//
	public InverseResult inverse(EarthEllipsoid ellipsoid, LatLng point1, LatLng point2) {
		
		Logger logger = GreatArcLogger.getLogger();
		logger.log(Level.FINER, "inverse() started.");

		double	lat1 = point1.getLatRad();		// 始点の緯度
		double	lng1 = point1.getLngRad();		// 始点の経度
		double	lat2 = point2.getLatRad();		// 到達点の緯度
		double	lng2 = point2.getLngRad();		// 到達点の経度
		double	A    = lng2 - lng1;				// 始点から到達点への経度差（東向きが正）
		double	B;								// 始点から到達点へ向けた方位角（北が０。時計回りが正）
		double	C;								// 到達点から始点へ向けた方位角（北が０。時計回りが正）
		double	a;								// 始点と到達点との中心角（半径を掛けると距離となる）
		double	b    = Math.PI / 2.0 - lat2;	// 北極と到達点との中心角（π／２　から到達点の緯度を減算したものに一致）
		double	c    = Math.PI / 2.0 - lat1;	// 北極と始点との中心角（π／２　から始点の緯度を減算したものに一致）
		
		// 扁平率が０以外のモデルが与えられた場合は、ワーニングを出す。
		{
			if(ellipsoid.getFlattening() != 0.0) {
				logger.log(Level.WARNING, "flattening of the given speroid is NOT 0.0");
			}
		}
		
		// まず２点間の距離を求める（中心角を求め、中心角に半径を乗算したものが距離となる）
		// cos(a) = cos(b) * cos(c) + sin(b) * sin(c) * cos(A) からaを求める。
		{
			a = Math.acos(  Math.cos(b) * Math.cos(c)
			              + Math.sin(b) * Math.sin(c) * Math.cos(A) );
			logger.log(Level.FINER, "a(rad) = " + a);
		}
		
		// 次に方位を求める(aは、↑で求まっている)
		// cos(b) = cos(c) * cos(a) + sin(c) * sin(a) * cos(B) からBを求める
		// cos(c) = cos(a) * cos(b) + sin(a) * sin(b) * cos(C) からCを求める
		{
			B = Math.acos(  (Math.cos(b) - Math.cos(c) * Math.cos(a)) 
			              / (Math.sin(c) * Math.sin(a))               );
			
			C = Math.acos(  (Math.cos(c) - Math.cos(a) * Math.cos(b))
			              / (Math.sin(a) * Math.sin(b))               );
			
			logger.log(Level.FINER, "A(deg) = " + Math.toDegrees(A));
			logger.log(Level.FINER, "B(deg) = " + Math.toDegrees(B));
			logger.log(Level.FINER, "C(deg) = " + Math.toDegrees(C));
			
			//
			// これだけじゃ足りない。。。acos()は、０～π（北～東～南）を返し、
			// π～２π（南～西～北）が分からないから。
			// 経度差がプラスで１８０°を超えていたら、実は始点から西向き
			//                           超えていなければ、到達点から西向き
			// 経度差がマイナスで－１８０°を超えていたら、実は到達点から西向き
			//                               超えていなければ、始点から西向き
			if(A >= 0.0) {
				if(A >= Math.PI)  { B = Math.PI * 2.0 - B; }
				else              { C = Math.PI * 2.0 - C; }
			} else {
				if(A <= -Math.PI) { C = Math.PI * 2.0 - C; }
				else              { B = Math.PI * 2.0 - B; }
			}
			
			logger.log(Level.FINER, "B(deg) = " + Math.toDegrees(B));
			logger.log(Level.FINER, "C(deg) = " + Math.toDegrees(C));
		}
		
		// 値を入れて返却する。
		InverseResult ret = new InverseResult();
		ret.distance          = a * ellipsoid.getAxis();	// 中心角（ラジアン）　ｘ　半径
		ret.azimuthFromPoint1 = B;
		ret.azimuthFromPoint2 = C;
		
		logger.log(Level.FINER, "inverse() terminated normally.");
		
		return ret;
	}
}