/**
 *  ある地点の緯度経度を保持するクラス
 *
 *  @author Toshi Tamamori
 */

//
//  ・ここでは、高度は保持しない。（当プログラムでは必要としないので）
//  ・度分秒（この際秒だけは小数）、ラジアン値を保持し、互いの整合を維持する。
//  ・設定（コンストラクタ）は、度（小数あり）と、度、分、秒（秒だけ小数あり）の２つを用意する。
//  ・緯度は北緯を正とする。
//  ・経度は東経を正とする。
//
//  以下、緯度の違いと地球表面上の距離の考察
//  緯度　　　　１分は、約１８５２。０メートル　（１国際海里）　　　０．０１６７　　度
//      　　　　１秒は、約　３０８．７メートル　　　　　　　　　　　０．０００２７８度
//      ０．１　　秒は、約　　３０．９メートル
//      ０．００１秒は、約　　　０．３メートル　　　　　　　　　　　０．００００００２７８度
//  とすると、
//  ０．００１秒は、度で表現すると（３６００で除算して）０．００００００２７８度なので、
//  度の単位で、小数点８か９桁までの精度で計算すれば、地球上の距離で１メートル未満の精度であると言える。
//  (既に上記にあるように秒で表現すれば、少数点３桁まで）
//
//  なお、１国際海里は、正確に１８５２メートルと定義されている。
//  上記で「約」を使ったのは、地球モデルによって緯度１分の距離は微妙に異なることから、
//  敢えて「約」を付けているものである。
//

public class LatLng {
	
	final private static double MAX_LATITUDE  =   90.0 ;
	final private static double MIN_LATITUDE  = (-90.0);
	final private static double MAX_LONGITUDE =   180.0 ;
	final private static double MIN_LONGITUDE = (-180.0);
	
	private double	latDeg; //度
	private double	lngDeg; //度
	private double	latRad; //ラジアン
	private double	lngRad; //ラジアン
	
	// constructors
	// コンストラクタには戻り値はありません。
	// さらに、戻り値がないからと言って voidキーワードを
	// 書くこともしません。（書くと変なことになります。）
	// わざと、引数なしのコンストラクタは用意せずに、必ず
	// オブジェクト生成時に緯度経度を指定させます。
	/**
	 * 指定された緯度経度でインスタンスを生成する。
	 *
	 * @param latDeg 緯度（度）北緯が正　-90.0 ～ 90.0
	 * @param lngDeg 経度（度）東経が正　-180.0 ～ 180.0
	 */
	public LatLng(double latDeg, double lngDeg) {
		setLatDeg(latDeg);
		setLngDeg(lngDeg);
	}
	
	/**
	 * 指定された緯度経度でインスタンスを生成する。
	 *
	 * @param latDegDeg 緯度　DMS表記の場合の度の部分　北緯が正(-90 ～ 90)
	 * @param latMin    　　　分の部分 (0 ～ 59)
	 * @param latSec    　　　秒の部分 (0.0 ～ 59.999...)
	 * @param lngDegDeg 経度　DMS表記の場合の度の部分　東経が正(-180 ～ +180)
	 * @param lngMin    　　　分の部分（0 ～ 59)
	 * @param lngSec    　　　秒の部分 (0.0 ～ 59.999...)
	 */
	public LatLng(int latDegDeg, int latMin, double latSec,
	              int lngDegDeg, int lngMin, double lngSec) {
		double latDeg = (double)latDegDeg + (double)latMin / 60.0 + latSec / 3600.0;
		double lngDeg = (double)lngDegDeg + (double)lngMin / 60.0 + lngSec / 3600.0;
		setLatDeg(latDeg);
		setLngDeg(lngDeg);
	}
	
	//
	//
	//
	private void setLatDeg(double latDeg) {
		if(isValidLat(latDeg)) {
			this.latDeg = latDeg;
			this.latRad = Math.toRadians(latDeg);
			return;
		} else {
			throw new IllegalArgumentException("illegal latitude value." + latDeg);
		}
	}
	
	//
	//
	//
	private void setLngDeg(double lngDeg) {
		if(isValidLng(lngDeg)) {
			this.lngDeg = lngDeg;
			this.lngRad = Math.toRadians(lngDeg);
			return;
		} else {
			throw new IllegalArgumentException("illegal longitude value." + lngDeg);
		}
	}
	
	/**
	 * 緯度経度の文字列（度およびN/S, E/W）
	 * 小数点６桁まで
	 * 
	 * @return 緯度経度を表現する文字列（e.g Lat = nn.ffffff N  Lng = nnn.ffffff W）
	 */
	public String toString() {
		char         ns;
		char         ew;
		
		if(0.0 <= latDeg) { ns = 'N'; } else { ns = 'S'; }
		if(0.0 <= lngDeg) { ew = 'E'; } else { ew = 'W'; }
		
		return String.format("Lat = %2.9f %c  Lng = %3.9f %c",
		                     Math.abs(latDeg), ns, Math.abs(lngDeg), ew);
	}
	
	/**
	 * 緯度経度の文字列（DMS表記およびN/S, E/W）
	 * 秒は小数点３桁まで
	 *
	 * @return 緯度経度を表現する文字列（e.g. Lat = nn°nn’nn"S Lng = nnn°nn’nn”E）
	 */
	public String toStringInDegMinSec() {
		char         ns;
		char         ew;
		double       latDegDeg;
		double       latMin;
		double       latSec;
		double       lngDegDeg;
		double       lngMin;
		double       lngSec;
		double       work;
		
		if(0.0 <= latDeg) { ns = 'N'; } else { ns = 'S'; }
		if(0.0 <= lngDeg) { ew = 'E'; } else { ew = 'W'; }
		
		work      = Math.abs(latDeg);
		latDegDeg = Math.floor(work);
		work      = (work - latDegDeg) * 60.0;
		latMin    = Math.floor(work);
		work      = (work - latMin)    * 60.0;
		latSec    = Math.floor(work);
		
		work      = Math.abs(lngDeg);
		lngDegDeg = Math.floor(work);
		work      = (work - lngDegDeg) * 60.0;
		lngMin    = Math.floor(work);
		work      = (work - lngMin)    * 60.0;
		lngSec    = Math.floor(work);
		
		return String.format("Lat = %2d°%2d’%2.3f” %c  Lng = %3d°%2d’%2.3f %c",
		                     (int)latDegDeg, (int)latMin, latSec, ns,
		                     (int)lngDegDeg, (int)lngMin, lngSec, ew                      );
	}
	
	/**
	 * 緯度（ラジアン値）の取得　（北緯が正）
	 *
	 * @return 緯度（ラジアン値。北緯が正。-π／２　～　π／２）
	 */
	public double getLatRad() {
		return latRad;
	}
	
	/**
	 * 経度（ラジアン値）の取得　（東経が正）
	 * @return 経度（ラジアン値。東経が正。-π　～　π）
	 */
	public double getLngRad() {
		return lngRad;
	}
	
	//
	//
	//
	private boolean isValidLat(double latDeg) {
		if(latDeg < MIN_LATITUDE) { return false; }
		if(MAX_LATITUDE < latDeg) { return false; }
		return true;
	}
	
	//
	//
	//
	private boolean isValidLng(double lngDeg) {
		if(lngDeg < MIN_LONGITUDE) { return false; }
		if(MAX_LONGITUDE < lngDeg) { return false; }
		return true;
	}
}
