import java.io.FileInputStream;
import java.io.IOException;
import java.util.Properties;
import java.util.logging.*;

/**
 * 大圏航路計算プログラム。
 * 
 * 計算に使用する準拠地球楕円体モデルとアルゴリズムは設定ファイル(GreatArc.properties)から読み取る。
 *
 * コマンドラインパラメータ:　後述している parseParametersのコメントを参照のこと。
 *
 * 一度の呼び出しでは一度の計算とし、複数の計算をさせるためには、周りの（例えばバッチファイルや
 * シェルスクリプトファイル）により、当プログラムを複数回呼び出すものとする。
 *
 * Java標準には、コマンドラインパラメータの解析用のユーティリティは存在していないようだ。
 * 従って、当該解析には、自作のソースコードとした。
 */

public class GreatArc {
	
	final private static String  propFileName = "./properties/GreatArc.properties.xml";
	private static EarthEllipsoid       ee;  // 使用している準拠地球楕円体を保持
	private static GreatArcAlg          gaa; // 使用している大圏航路計算アルゴリズムを保持
	private static boolean              isDirect;
	private static LatLng               p1;
	private static LatLng               p2;
	private static double               distance;  // 単位：メートル
	private static double               az1to2;    // 単位：ラジアン
	private static double               az2to1;    // 単位：ラジアン
	
	/**
	 * 当クラスをインスタンス化することを意図していません。
	 */
	private GreatArc() { }

	/**
	 * メインプログラム
	 *
	 * @param argv 説明を省略しています
	 */
	public static void main(String[] argv) {
		
		// まっ先にロガーの設定
		//
		Logger logger = GreatArcLogger.getLogger();
		logger.log(Level.INFO, "■■■ GreatArc program started. ■■■");
		
		// 設定ファイル読み込みと指定されたモデル・アルゴリズムの保持
		//
		loadProperties();
		
		// 与えられた引数（コマンドラインパラメータ）を取得する
		//
		parseParameters(argv);
		
		// 計算する。
		//
		calculation();
		
		// 出力する。
		printResult();
		
		// プログラムを終了する
		//
		logger.log(Level.INFO, "■■■ GreatArc program terminated. ■■■");
		return;
	}
	
	//
	// 設定ファイルから設定値を読み込み、それらのクラスを生成して、
	// 静的メンバ変数に保持する。
	//
	private static void loadProperties() {
		
		Logger logger = GreatArcLogger.getLogger();
		
		logger.log(Level.FINER, "loadProperties() started.");
		
		String ellipsoidClassName = null;
		String algorithmClassName = null;
		
		// 設定ファイルの読み込み
		try {
			Properties      settings = new Properties();
			FileInputStream inStream = new FileInputStream(propFileName);
			
			settings.loadFromXML(inStream);
			ellipsoidClassName = settings.getProperty("ellipsoid");
			algorithmClassName = settings.getProperty("algorithm");
			inStream.close();
			
			logger.log(Level.FINER, "  ellipsoid : " + ellipsoidClassName);
			logger.log(Level.FINER, "  algorithm : " + algorithmClassName);
			
		} catch (Exception e) {
			logger.log(Level.SEVERE, "CANNOT load settings. exception occurred.");
			logger.log(Level.SEVERE, e.toString());
		}
		
		// クラス名文字列に合致する具象クラスのインスタンスを生成し、インタフェースに代入する。
		// （そのあと本プログラムで当該インタフェースを一貫して使用することとし、
		//   具象クラスをそのまま使うことはしない）
		try {
			Class<?> cls = Class.forName(ellipsoidClassName);
			// ee = (EarthEllipsoid)obj;
			// ee = cls.getDeclaredConstructor().newInstance(); // こうはいきなり書けないみたいだ。
			// Object   obj = cls.newInstance(); // これは非推奨のAPIを使用しているとの警告が出る。
			Object obj = cls.getDeclaredConstructor().newInstance();
			ee = (EarthEllipsoid)obj;
			logger.log(Level.INFO, "EarthEllipsoid is set to " + ellipsoidClassName);
		} catch(Exception e) {
			logger.log(Level.SEVERE, "CANNOT instantiate EarthEllisoid.");
			logger.log(Level.SEVERE, e.toString());
		}
		
		try {
			Class<?> cls = Class.forName(algorithmClassName);
			Object obj = cls.getDeclaredConstructor().newInstance();
			gaa = (GreatArcAlg)obj;
			logger.log(Level.INFO, "GreatArcAlg is set to " + algorithmClassName);
		} catch(Exception e) {
			logger.log(Level.SEVERE, "CANNOT instatiate GreatArcAlg.");
			logger.log(Level.SEVERE, e.toString());
		}
		
		logger.log(Level.FINER, "loadProperties() terminated normally.");
		return;
	}
	
	// コマンドラインパラメータを解析して、クラス変数に格納する。
	// 第一引数　direct | inverse  必須 direct問題かinverse問題かを指定する
	// 第二引数　sdd.fffff         必須 １点目の緯度（度で小数を含めた表記）
	// 第三引数　sddd.fffff        必須 １点目の経度（度で小数を含めた表記）
	// [第一引数がdirectの場合]
	// 第四引数　dddddd.ffffff     必須 １点目からの距離（メートル）
	// 第五引数　ddd.fffff         必須 １点目からの方位角（度で小数を含めた表記）
	// [第一引数がinverseの場合]
	// 第四引数　sdd.fffff         必須 ２点目の緯度（度で小数を含めた表記）
	// 第五引数　sddd.ffff         必須 ２点目の緯度（度で小数を含めた表記）
	// 
	// private static EarthEllipsoid       ee;  // 使用している準拠地球楕円体を保持
	// private static GreatArcAlg          gaa; // 使用している大圏航路計算アルゴリズムを保持
	// private static boolean              isDirect;
	// private static LatLng               p1;
	// private static LatLng               p2;
	// private static double               distance;  // 単位：メートル
	// private static double               az1to2;    // 単位：ラジアン
	// private static double               az2to1;    // 単位：ラジアン
	//
    private static void parseParameters(String[] argv) throws IllegalArgumentException {
		
		Logger logger = GreatArcLogger.getLogger();
		
		logger.log(Level.FINER, "parseParameters() started.");
		
		for(int loop=0; loop<argv.length; loop++) {
			logger.log(Level.FINER, " param " + loop + " : "+ argv[loop]);
		}
		
		// directでもinverseでも、引数は５つでなければならない。
		if(argv.length < 5) {
			logger.log(Level.SEVERE, "invalid parameters. should specify 5 parameters.");
			throw new IllegalArgumentException("invalid count of parameters.");
		}
		
		// 第一引数の取得
		if(argv[0].equals("direct"))       { isDirect = true;  }
		else if(argv[0].equals("inverse")) { isDirect = false; }
		else {
			logger.log(Level.SEVERE, "invalid parameters. specified problem kind (" + argv[0] + ")");
			throw new IllegalArgumentException("invalid problem kind");
		}
		
		// 第二引数・第三引数の取得
		try {
			double p1lat = Double.parseDouble(argv[1]);
			double p1lng = Double.parseDouble(argv[2]);
			
			if((p1lat < -90.0) || (90.0 < p1lat)) {
				logger.log(Level.SEVERE, "specified latitude range (" + p1lat + ")");
				throw new IllegalArgumentException("latitude range.");
			}
			
			if((p1lng < -180.0) || (180.0 < p1lng)) {
				logger.log(Level.SEVERE, "specified longitude range (" + p1lng + ")");
				throw new IllegalArgumentException("longitude range.");
			}
			
			p1 = new LatLng(p1lat, p1lng);
			
		} catch(Exception e) {
			logger.log(Level.SEVERE, "invalid position of point 1.");
			throw e;
		}
		
		// 第一引数が directの場合の第四引数・第五引数の取得
		if(isDirect) {
			try {
				double dis = Double.parseDouble(argv[3]);
				double az  = Double.parseDouble(argv[4]);
				
				if(dis < 0.0) {
					logger.log(Level.SEVERE, "distance should be a positive value. (", + dis + ")");
					throw new IllegalArgumentException("distance range.");
				}
				
				if((az < 0.0) || (360.0 < az)) {
					logger.log(Level.SEVERE, "azimuth should be 0.0 - 360.0. (", + az + ")");
					throw new IllegalArgumentException("azimuth range.");
				}
				
				distance = dis;
				az1to2   = Math.toRadians(az);
				
			} catch(Exception e) {
				logger.log(Level.SEVERE, "invalid distance and/or azimuth.");
				throw e;
			}
			
		} else { // inverseの場合の第四・第五引数の取得
			try {
				double p2lat = Double.parseDouble(argv[3]);
				double p2lng = Double.parseDouble(argv[4]);
				
				if((p2lat < -90.0) || (90.0 < p2lat)) {
					logger.log(Level.SEVERE, "specified latitude range (" + p2lat + ")");
					throw new IllegalArgumentException("latitude range.");
				}
				
				if((p2lng < -180.0) || (180.0 < p2lng)) {
					logger.log(Level.SEVERE, "specified longitude range (" + p2lng + ")");
					throw new IllegalArgumentException("longitude range.");
				}
				
				p2 = new LatLng(p2lat, p2lng);
				
			} catch(Exception e) {
				logger.log(Level.SEVERE, "invalid position of point 2.");
				throw e;
			}
		}
		
		logger.log(Level.FINER, "parseParameters() terminated normally.");
		return;
	}
	
	
	// 計算する
	// コマンドラインにて与えられたパラメータは、当クラスのクラス変数に格納されている。
	// 計算結果は、その他のクラス変数に格納する。
	//
    private static void calculation() {
		
		if(isDirect) {
			DirectResult result = gaa.direct(ee, p1, distance, az1to2);
			
			p2 = result.pointReached;
			az2to1 = result.azimuthFromPoint2;
		}
		else {
			InverseResult result = gaa.inverse(ee, p1, p2);
			
			distance = result.distance;
			az1to2   = result.azimuthFromPoint1;
			az2to1   = result.azimuthFromPoint2;
		}
		
		return;
	}
	
	
	// 結果を標準出力にプリントする（ログにも書きだす）
	//
	// ここで、方位角の精度と地球表面上の距離の精度の考察
	// ズレの話なので、微小な角度で考えることとする。
	// すると十分微小な角度においては、角度と、円弧の長さはほぼ等しいと考えて良い。
	// （ただし、この角度とは度数表記ではなく、弧度法によるラジアン値であることに注意）
	//           円弧の長さ＝半径×角度（ラジアン値）
	// さて、このプログラムの主題としているのは、地球上の距離、方位角であることから、
	// 最大でも地球半周分の距離を考えることとし、それは約20,000kmである。
	// 計算してみると、角度１秒　　　　のズレは、20,000km先で、約97メートルズレる。
	//                 角度０．００１秒のズレは、20,000km先で、約97ミリメートルズレる。
	// 従って、方位角の精度として、秒で、小数点第三位までの精度であれば、十分な精度と結論づける。
	// 度に換算すると、3600で除算して、小数点第七位までで、〃
	//
	private static void printResult() {
		
		Logger logger = GreatArcLogger.getLogger();
		String disStr = String.format("%,.3f", distance);
		String az1to2Str = String.format("%3.7f", Math.toDegrees(az1to2));
		String az2to1Str = String.format("%3.7f", Math.toDegrees(az2to1));
		
		logger.log(Level.INFO, " point 1 : " + p1.toString());
		logger.log(Level.INFO, " point 2 : " + p2.toString());
		logger.log(Level.INFO, " dis(m)     = " + disStr);
		logger.log(Level.INFO, " az1to2(deg)= " + az1to2Str);
		logger.log(Level.INFO, " az2to1(deg)= " + az2to1Str);
		
		System.out.println(" point 1 : " + p1.toString());
		System.out.println(" point 2 : " + p2.toString());
		System.out.println(" dis(m)     = " + disStr);
		System.out.println(" az1to2(deg)= " + az1to2Str);
		System.out.println(" az2to1(deg)= " + az2to1Str);

		return;
	}
}